/* squiral --- squiraling worms */

#if 0
static const char sccsid[] = "@(#)squiral.c  5.87 2025/12/18 xlockmore";
#endif

/*-
 * squiral, by "Jeff Epler" <jepler AT inetnebr.com>, 18-mar-1999.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#ifdef STANDALONE
#define MODE_squiral
#define DEFAULTS "*delay:	10000 \n" \
  "*fpsSolid:	true  \n" \
  "*count:	0 \n" \
  "*ncolors:	100 \n" \

#define free_squiral 0
#define UNIFORM_COLORS
#include "xlockmore.h"	/* in xscreensaver distribution */
#else /* STANDALONE */
#include "xlock.h"	/* in xlockmore distribution */
#include "color.h"
#include "iostuff.h"
#endif /* STANDALONE */
#include "automata.h"

#ifdef MODE_squiral

#define FLOATRAND ((double) LRAND() / ((double) MAXRAND))
#define R(x)  NRAND(x)
#define PROB(x) (FLOATRAND < (x))

/* 0- 3 left-winding  */
/* 4- 7 right-winding */

#define DEF_DISORDER "0.005"
#define DEF_HANDEDNESS  "0.5" /* no preference */
#define DEF_SCALE  "1" /* smallest possible */
#define DEF_CYCLE "False"
#define DEF_FILL  "0.75"
#define DEF_NEIGHBORS  "0" /* choose random value */
#define DEF_VERTICAL "False"

static float disorder;
static float handedness;
static int  scale;
static float fill;
static Bool cycle;
static int  neighbors;
static Bool vertical;

static XrmOptionDescRec opts[] =
{
	{(char *) "-disorder", (char *) ".squiral.disorder", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-handedness", (char *) ".squiral.handedness", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-scale", (char *) ".squiral.scale", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-fill", (char *) ".squiral.fill", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-cycle", (char *) ".squiral.cycle", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+cycle", (char *) ".squiral.cycle", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-neighbors", (char *) ".squiral.neighbors", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-vertical", (char *) ".squiral.vertical", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+vertical", (char *) ".squiral.vertical", XrmoptionNoArg, (caddr_t) "off"}
};
static argtype vars[] =
{
	{(void *) & disorder, (char *) "disorder", (char *) "Disorder", (char *) DEF_DISORDER, t_Float},
	{(void *) & handedness, (char *) "handedness", (char *) "Handedness", (char *) DEF_HANDEDNESS, t_Float},
	{(void *) & scale, (char *) "scale", (char *) "Scale", (char *) DEF_SCALE, t_Int},
	{(void *) & fill, (char *) "fill", (char *) "Fill", (char *) DEF_FILL, t_Float},
	{(void *) & cycle, (char *) "cycle", (char *) "Cycle", (char *) DEF_CYCLE, t_Bool},
	{(void *) & neighbors, (char *) "neighbors", (char *) "Neighbors", (char *) DEF_NEIGHBORS, t_Int},
	{(void *) & vertical, (char *) "vertical", (char *) "Vertical", (char *) DEF_VERTICAL, t_Bool}
};
static OptionStruct desc[] =
{
	{(char *) "-disorder num", (char *) "chance of disorder"},
	{(char *) "-handedness num", (char *) "prefer left or right"},
	{(char *) "-scale num", (char *) "size of boxes"},
	{(char *) "-/+cycle", (char *) "cycle the colors"},
	{(char *) "-neighbors num", (char *) "connections of 3, 4, 5, and 6"},
	{(char *) "-/+vertical", (char *) "change orientation for hexagons and triangles"}
};

ENTRYPOINT ModeSpecOpt squiral_opts =
{sizeof opts / sizeof opts[0], opts, sizeof vars / sizeof vars[0], vars, desc};

#ifdef USE_MODULES
const ModStruct squiral_description =
{"squiral", "init_squiral", "draw_squiral", "release_squiral",
 "init_squiral", "init_squiral", (char *) NULL, &squiral_opts,
 10000, 0, 50000, -3, 64, 1.0, "",
 "Shows spiral-producing automata", 0, NULL};

#endif

typedef struct {
	int h;
	int v;
	int s;
	int c;
	int cc;
} wormstruct;

typedef struct {
	int width, height, count, cycle;
	double frac, disorder, handedness;
	int ncolors;

	int delay;

	int cov;
	int dirh[4];
	int dirv[4];

	int *fill;

	wormstruct *worms;
	int inclear;
	int scale;
	int neighbors, polygon;
	int vertical;
	int xs, ys, xb, yb, nrows, ncols;
	union {
		XPoint hexagon[7];
		XPoint triangle[2][4];
		XPoint pentagon[4][6];
	} shape;
} squiralstruct;

static squiralstruct *squirals = (squiralstruct *) NULL;


#define CLEAR1(x,y) (!st->fill[((y)%st->height)*st->width+(x)%st->width])
#define MOVE1(x,y) (st->fill[((y)%st->height)*st->width+(x)%st->width]=1, \
	XFillRectangle (display, window, MI_GC(mi), \
		((x) % st->width) * st->scale, \
		((y) % st->height) * st->scale, \
		st->scale, st->scale), \
		st->cov++)

#define CLEARDXY(x,y,dx,dy) CLEAR1(x+dx, y+dy) && CLEAR1(x+dx+dx, y+dy+dy)
#define MOVEDXY(x,y,dx,dy) MOVE1 (x+dx, y+dy), MOVE1(x+dx+dx, y+dy+dy)

#define CLEAR(d) CLEARDXY(w->h,w->v, st->dirh[d],st->dirv[d])
#define MOVE(d) (MOVEDXY(w->h,w->v, st->dirh[d],st->dirv[d]), \
	w->h=w->h+st->dirh[d]*2, \
	w->v=w->v+st->dirv[d]*2, dir=d)

#define RANDOM (void) (w->h = R(st->width), \
	w->v = R(st->height), \
	w->c = R(st->ncolors), \
	type=R(2), \
	dir=R(4), \
	(st->cycle && (w->cc=R(3)+st->ncolors)))


#define SUCC(x) ((x+1)%4)
#define PRED(x) ((x+3)%4)
#define CCW	PRED(dir)
#define CW	SUCC(dir)
#define REV	((dir+2)%4)
#define STR	(dir)
#define TRY(x)	if (CLEAR(x)) { MOVE(x); break; }

#if 0
  /* TODO: Add in for future grids... */
#define MINGRIDSIZE 24
#define MINSIZE 4

static void
drawCell(ModeInfo * mi, int col, int row, unsigned char state)
{
	squiralstruct *st = &squirals[MI_SCREEN(mi)];
	Display *display = MI_DISPLAY(mi);
	Window window = MI_WINDOW(mi);
	GC gc = MI_GC(mi);

	if (st->neighbors == 6) {
		int ccol = 2 * col + !(row & 1), crow = 2 * row;

		if (st->vertical) {
			st->shape.hexagon[0].x = st->xb + ccol * st->xs;
			st->shape.hexagon[0].y = st->yb + crow * st->ys;
		} else {
			st->shape.hexagon[0].y = st->xb + ccol * st->xs;
			st->shape.hexagon[0].x = st->yb + crow * st->ys;
		}
		if (st->xs == 1 && st->ys == 1)
			XDrawPoint(display, window, gc,
				st->shape.hexagon[0].x,
				st->shape.hexagon[0].y);
		else
			XFillPolygon(display, window, gc,
				st->shape.hexagon, 6,
				Convex, CoordModePrevious);
	} else if (st->neighbors == 4) {
		XFillRectangle(display, window, gc,
			st->xb + st->xs * col, st->yb + st->ys * row,
			st->xs - (st->xs > 3), st->ys - (st->ys > 3));
	} else if (st->neighbors == 5) {
		int map[4] = {2, 0, 1, 3};
		int orient = ((row & 1) * 2 + col) % 4;
		int offsetX = 0, offsetY = 0;
		switch (orient) {
		case 0: /* up */
			offsetX = st->xs;
			break;
		case 1: /* down */
			offsetY = 3 * st->ys / 2 - 1;
			break;
		case 2: /* left */
			offsetX = -st->xs / 2;
			offsetY = 3 * st->ys / 4;
			break;
		case 3: /* right */
			offsetX = 3 * st->xs / 2 - 1;
			offsetY = 3 * st->ys / 4;
			break;
		default:
			(void) printf("wrong orient %d\n", orient);
		}
		orient = map[orient];
		st->shape.pentagon[orient][0].x = st->xb +
			col * st->xs + offsetX;
		st->shape.pentagon[orient][0].y = st->yb +
			row * st->ys + offsetY;
		if (st->xs <= 2 || st->ys <= 2)
			XDrawPoint(display, window, gc,
				st->shape.pentagon[orient][0].x,
				st->shape.pentagon[orient][0].y);
		else
			XFillPolygon(display, window, gc,
				st->shape.pentagon[orient], 5, Convex, CoordModePrevious);
	} else {		/* TRI */
		int orient = (col + row) % 2; /* O left 1 right */
		Bool small = (st->xs <= 3 || st->ys <= 3);

		if (st->vertical) {
			st->shape.triangle[orient][0].x = st->xb + col * st->xs;
			st->shape.triangle[orient][0].y = st->yb + row * st->ys;
			if (small)
				st->shape.triangle[orient][0].x +=
					((orient) ? -1 : 1);
			else
				st->shape.triangle[orient][0].x +=
					(st->xs / 2 - 1) * ((orient) ? 1 : -1);
		} else {
			st->shape.triangle[orient][0].y = st->xb + col * st->xs;
			st->shape.triangle[orient][0].x = st->yb + row * st->ys;
			if (small)
				st->shape.triangle[orient][0].y +=
					((orient) ? -1 : 1);
			else
				st->shape.triangle[orient][0].y +=
					(st->xs / 2 - 1) * ((orient) ? 1 : -1);
		}
		if (small)
			XDrawPoint(display, window, gc,
				st->shape.triangle[orient][0].x,
				st->shape.triangle[orient][0].y);
		else {
			XFillPolygon(display, window, gc,
				st->shape.triangle[orient], 3,
				Convex, CoordModePrevious);
			XSetForeground(display, gc, MI_BLACK_PIXEL(mi));
			XDrawLines(display, window, gc,
				st->shape.triangle[orient], 4,
				CoordModePrevious);
		}
	}
}

static void
initCell(ModeInfo * mi)
{
	squiralstruct *st = &squirals[MI_SCREEN(mi)];
	int size = MI_SIZE(mi);
	st->width = MI_WIDTH(mi);
	st->height = MI_HEIGHT(mi);
	if (st->neighbors == 6) {
		int nccols, ncrows, side;

		st->polygon = 6;
		if (!st->vertical) {
			st->height = MI_WIDTH(mi);
			st->width = MI_HEIGHT(mi);
		}
		if (st->width < 8)
			st->width = 8;
		if (st->height < 8)
			st->height = 8;
		if (size < -MINSIZE)
			st->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		else if (size < MINSIZE) {
			if (!size)
				st->ys = MAX(MINSIZE, MIN(st->width, st->height) / MINGRIDSIZE);
			else
				st->ys = MINSIZE;
		} else
			st->ys = MIN(size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE));
		st->xs = st->ys;
		nccols = MAX(st->width / st->xs - 2, 2);
		ncrows = MAX(st->height / st->ys - 1, 4);
		st->ncols = nccols / 2;
		st->nrows = 2 * (ncrows / 4);
		st->xb = (st->width - st->xs * nccols) / 2 + st->xs / 2;
		st->yb = (st->height - st->ys * (ncrows / 2) * 2) / 2 +
			st->ys / 3;
		for (side = 0; side < 6; side++) {
			if (st->vertical) {
				st->shape.hexagon[side].x =
					(st->xs - 1) * hexagonUnit[side].x;
				st->shape.hexagon[side].y =
					((st->ys - 1) * hexagonUnit[side].y /
					2) * 4 / 3;
			} else {
				st->shape.hexagon[side].y =
					(st->xs - 1) * hexagonUnit[side].x;
				st->shape.hexagon[side].x =
					((st->ys - 1) * hexagonUnit[side].y /
					2) * 4 / 3;
			}
		}
	} else if (st->neighbors == 4) {
		st->polygon = 4;
		if (size < -MINSIZE)
			st->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		else if (size < MINSIZE) {
			if (!size)
				st->ys = MAX(MINSIZE, MIN(st->width, st->height) / MINGRIDSIZE);
			else
				st->ys = MINSIZE;
		} else
			st->ys = MIN(size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE));
		st->xs = st->ys;
		st->ncols = MAX(st->width / st->xs, 2);
		st->nrows = MAX(st->height / st->ys, 2);
		st->xb = (st->width - st->xs * st->ncols) / 2;
		st->yb = (st->height - st->ys * st->nrows) / 2;
	} else if (st->neighbors == 5) {
				int orient, side;
		st->polygon = 5;
		if (st->width < 2)
		if (st->height < 2)
			st->height = 2;
		if (size < -MINSIZE)
			st->xs = NRAND(MIN(-size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		else if (size < MINSIZE) {
			if (!size) {
				int min = MIN(st->width, st->height) / (4 * MINGRIDSIZE);
				int max = MIN(st->width, st->height) / (MINGRIDSIZE);

				st->xs = MAX(MINSIZE, min + NRAND(max - min + 1));
			} else
				st->xs = MINSIZE;
		} else
			st->xs = MIN(size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE));
		st->ys = st->xs * 2;
		st->ncols = (MAX((st->width - 4) / st->xs, 8) / 4) * 4;
		st->nrows = (MAX((st->height - st->ys / 2) / st->ys, 8)) / 2 * 2;
		st->xb = (st->width - st->xs * st->ncols) / 2;
		st->yb = (st->height - st->ys * st->nrows) / 2 - 2;
		for (orient = 0; orient < 4; orient++) {
			for (side = 0; side < 5; side++) {
				st->shape.pentagon[orient][side].x =
					2 * (st->xs - 1) * pentagonUnit[orient][side].x / 4;
				st->shape.pentagon[orient][side].y =
					(st->ys - 1) * pentagonUnit[orient][side].y / 4;
			}
		}
	} else {		/* TRI */
		int orient, side;

		st->polygon = 3;
		if (!st->vertical) {
			st->height = MI_WIDTH(mi);
			st->width = MI_HEIGHT(mi);
		}
		if (st->width < 2)
			st->width = 2;
		if (st->height < 2)
			st->height = 2;
		if (size < -MINSIZE)
			st->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		else if (size < MINSIZE) {
			if (!size)
				st->ys = MAX(MINSIZE, MIN(st->width, st->height) / MINGRIDSIZE);
			else
				st->ys = MINSIZE;
		} else
			st->ys = MIN(size, MAX(MINSIZE, MIN(st->width, st->height) /
				MINGRIDSIZE));
		st->xs = (int) (1.52 * st->ys);
		st->ncols = (MAX(st->width / st->xs - 1, 2) / 2) * 2;
		st->nrows = (MAX(st->height / st->ys - 1, 2) / 2) * 2;
		st->xb = (st->width - st->xs * st->ncols) / 2 + st->xs / 2;
		st->yb = (st->height - st->ys * st->nrows) / 2 + st->ys / 2;
		for (orient = 0; orient < 2; orient++) {
			for (side = 0; side < 3; side++) {
				if (st->vertical) {
					st->shape.triangle[orient][side].x =
						(st->xs - 2) * triangleUnit[orient][side].x;
					st->shape.triangle[orient][side].y =
						(st->ys - 2) * triangleUnit[orient][side].y;
				} else {
					st->shape.triangle[orient][side].y =
						(st->xs - 2) * triangleUnit[orient][side].x;
				}
			}
		}
	}
}
#endif

static void
do_worm(ModeInfo * mi, squiralstruct *st, wormstruct *w)
{
	Display	*display = MI_DISPLAY(mi);
	Window window = MI_WINDOW(mi);
	int type = w->s / 4;
	int dir = w->s % 4;

	w->c = (w->c+w->cc) % st->ncolors;

	if (PROB(st->disorder))
		type=PROB(st->handedness);
	if (MI_NPIXELS(mi) >= 2) {
		XSetForeground(display, MI_GC(mi), MI_PIXEL(mi, w->c));
	} else {
		XSetForeground(display, MI_GC(mi), MI_WHITE_PIXEL(mi));
	}
	switch(type) {
	case 0: /* CCW */
 		TRY(CCW)
		TRY(STR)
		TRY(CW)
		RANDOM;
		break;
	case 1: /* CW */
		TRY(CW)
		TRY(STR)
		TRY(CCW)
		RANDOM;
		break;
	}
	w->s = type * 4 + dir;
	w->h = w->h % st->width;
	w->v = w->v % st->height;
}

static void
squiral_reset(squiralstruct *st)
{
	int i;
	if (st->worms)
		free(st->worms);
	if (st->fill)
		free(st->fill);

	st->worms=calloc(st->count, sizeof(wormstruct));
	st->fill=calloc(st->width * st->height, sizeof(int));

	st->dirh[0] = 0;
	st->dirh[1] = 1;
	st->dirh[2] = 0;
	st->dirh[3] = st->width - 1;
	st->dirv[0] = st->height - 1;
	st->dirv[1] = 0;
	st->dirv[2] = 1;
	st->dirv[3] = 0;
	for (i = 0; i < st->count; i++) {
		st->worms[i].h = R(st->width);
		st->worms[i].v = R(st->height);
		st->worms[i].s = R(4) + 4 * PROB(st->handedness);
		st->worms[i].c = R(st->ncolors);
		if (st->cycle)
			st->worms[i].cc = R(3) + st->ncolors;
		else
			st->worms[i].cc = 0;
	}
}

static void
free_squiral_screen(squiralstruct *st)
{
	if (st == NULL)
		return;
	if (st->worms)
		free(st->worms);
	if (st->fill)
		free(st->fill);
	st = NULL;
}

ENTRYPOINT void
init_squiral(ModeInfo * mi)
{
	Display	*display = MI_DISPLAY(mi);
	Window window = MI_WINDOW(mi);
	squiralstruct *st;
	XWindowAttributes xgwa;

	MI_INIT(mi, squirals);
	st = &squirals[MI_SCREEN(mi)];

	MI_CLEARWINDOW(mi);
	XGetWindowAttributes(display, window, &xgwa);

	st->scale = scale;
	if (xgwa.width > 2560 || xgwa.height > 2560)
		st->scale *= 3; /* Retina displays */

	st->width = xgwa.width / st->scale;
	st->height = xgwa.height / st->scale;

	st->count = MI_COUNT(mi);
	st->cycle = cycle;
	st->frac = fill;
	st->disorder = disorder;
	st->handedness = handedness;
	st->ncolors = MI_NPIXELS(mi);

	if (st->frac < 0.01)
		st->frac = 0.01;
	if (st->frac > 0.99)
		st->frac = 0.99;
	if (st->count == 0)
		st->count = (st->width + st->height) / 64;
	if (st->count < 1)
		st->count = 1;
	if (st->count > 1000)
		st->count = 1000;

	st->worms = NULL;
	st->fill = NULL;

	squiral_reset(st);
}

ENTRYPOINT void
draw_squiral(ModeInfo * mi)
{
	Display	*display = MI_DISPLAY(mi);
	Window window = MI_WINDOW(mi);
	squiralstruct *st;
	int i;

	if (squirals == NULL)
		return;
	st = &squirals[MI_SCREEN(mi)];

	if (st->inclear < st->height) {
		XSetForeground(display, MI_GC(mi), MI_BLACK_PIXEL(mi));
		XFillRectangle(display, window, MI_GC(mi),
			0,
			st->inclear * st->scale,
			(st->width - 1) * st->scale,
			st->scale);
		memset(&st->fill[st->inclear * st->width], 0,
			sizeof(int) * st->width);
		XFillRectangle(display, window, MI_GC(mi),
			0,
			(st->height-st->inclear - 1) * st->scale,
			(st->width - 1) * st->scale,
			st->scale);
		memset(&st->fill[(st->height - st->inclear - 1) * st->width], 0,
			sizeof(int) * st->width);
		st->inclear++;
		XFillRectangle(display, window, MI_GC(mi),
			0,
			st->inclear * st->scale,
			(st->width - 1) * st->scale,
			st->scale);
		if (st->inclear < st->height)
			memset(&st->fill[st->inclear * st->width], 0,
				sizeof(int)*st->width);
		XFillRectangle(display, window, MI_GC(mi),
			0,
			(st->height - st->inclear - 1) * st->scale,
			(st->width - 1) * st->scale, 
			st->scale);
		if (st->height - st->inclear >= 1)
			memset(&st->fill[(st->height - st->inclear - 1) * st->width], 0,
				sizeof(int) * st->width);
		st->inclear++;
		if (st->inclear > st->height / 2)
			st->inclear = st->height;
	} else if (st->cov > (st->frac * st->width * st->height)) {
		st->inclear = 0;
		st->cov = 0;
	}
	for (i = 0; i < st->count; i++)
		do_worm(mi, st, &st->worms[i]);
}

#ifdef STANDALONE
ENTRYPOINT void
reshape_squiral(ModeInfo *mi, int width, int height)
{
	Display	*display = MI_DISPLAY(mi);
	Window window = MI_WINDOW(mi);
	squiralstruct *st;

	if (squirals == NULL)
		return;
	st = &squirals[MI_SCREEN(mi)];

	st->scale = scale;
	if (width > 2560 || height > 2560)
		st->scale *= 3; /* Retina displays */

	st->width = width / st->scale;
	st->height = height / st->scale;
	squiral_reset(st);
	XClearWindow(display, window);
}

ENTRYPOINT Bool
squiral_handle_event(ModeInfo *mi, XEvent *event)
{
	squiralstruct *st = &squirals[MI_SCREEN(mi)];
	if (screenhack_event_helper(MI_DISPLAY(mi), MI_WINDOW(mi), event)) {
		squiral_reset(st);
		MI_CLEARWINDOW(mi);
		return True;
	}
	return False;
}
#endif

ENTRYPOINT void
release_squiral(ModeInfo * mi)
{
	if (squirals != NULL) {
		int screen;

		for (screen = 0; screen < MI_NUM_SCREENS(mi); screen++) {
			free_squiral_screen(&squirals[screen]);
		}
		free(squirals);
		squirals = (squiralstruct *) NULL;
	}
}

XSCREENSAVER_MODULE ("Squiral", squiral)

#endif /* MODE_squiral */

